<?php
/**
 * Copyright 2006 - 2013 TubePress LLC (http://tubepress.org)
 *
 * This file is part of TubePress (http://tubepress.org)
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

/**
 * Generates HTML for the embedded video player.
 */
class tubepress_impl_embedded_DefaultEmbeddedPlayerHtmlGenerator implements tubepress_spi_embedded_EmbeddedHtmlGenerator
{
    /**
     * @var ehough_epilog_api_ILogger Logger.
     */
    private $_logger;

    public function __construct()
    {
        $this->_logger = ehough_epilog_api_LoggerFactory::getLogger('Default Embedded Player HTML Generator');
    }

    /**
     * Spits back the text for this embedded player
     *
     * @param string $videoId The video ID to display
     *
     * @return string The text for this embedded player, or null if there was a problem.
     */
    public final function getHtml($videoId)
    {
        $embeddedPlayer = $this->_getEmbeddedPlayer($videoId);

        if ($embeddedPlayer === null) {

            $this->_logger->warn('Could not generate the embedded player HTML for ' . $videoId);

            return null;
        }

        $themeHandler           = tubepress_impl_patterns_sl_ServiceLocator::getThemeHandler();
        $eventDispatcherService = tubepress_impl_patterns_sl_ServiceLocator::getEventDispatcher();

        $template = $embeddedPlayer->getTemplate($themeHandler);

        $dataUrl            = $embeddedPlayer->getDataUrlForVideo($videoId);
        $embeddedPlayerName = $embeddedPlayer->getName();
        $providerName       = $embeddedPlayer->getHandledProviderName();

        /**
         * Build the embedded template event.
         */
        $embeddedTemplateEvent = new tubepress_api_event_TubePressEvent(

            $template,
            array(
                'videoId'                    => $videoId,
                'providerName'               => $providerName,
                'dataUrl'                    => $dataUrl,
                'embeddedImplementationName' => $embeddedPlayerName)
        );

        /**
         * Dispatch the embedded template event.
         */
        $eventDispatcherService->dispatch(

            tubepress_api_const_event_CoreEventNames::EMBEDDED_TEMPLATE_CONSTRUCTION,
            $embeddedTemplateEvent
        );

        /**
         * Pull the template out of the event.
         */
        $template = $embeddedTemplateEvent->getSubject();

        /**
         * Build the embedded HTML event.
         */
        $embeddedHtmlEvent = new tubepress_api_event_TubePressEvent(

            $template->toString(),
            array(
                'videoId'                    => $videoId,
                'providerName'               => $providerName,
                'dataUrl'                    => $dataUrl,
                'embeddedImplementationName' => $embeddedPlayerName)
        );

        /**
         * Dispatche the embedded HTML event.
         */
        $eventDispatcherService->dispatch(

            tubepress_api_const_event_CoreEventNames::EMBEDDED_HTML_CONSTRUCTION,
            $embeddedHtmlEvent
        );

        return $embeddedHtmlEvent->getSubject();
    }

    /**
     * @param $videoId
     *
     * @return tubepress_spi_embedded_PluggableEmbeddedPlayerService
     */
    private function _getEmbeddedPlayer($videoId)
    {
        $executionContext            = tubepress_impl_patterns_sl_ServiceLocator::getExecutionContext();
        $embeddedPlayers             = tubepress_impl_patterns_sl_ServiceLocator::getEmbeddedPlayers();
        $requestedEmbeddedPlayerName = $executionContext->get(tubepress_api_const_options_names_Embedded::PLAYER_IMPL);
        $recognizingProvider         = $this->_findProviderThatRecognizesVideoId($videoId);

        /**
         * The user has requested a specific embedded player that is registered. Let's see if the provider agrees.
         */
        if ($requestedEmbeddedPlayerName !== tubepress_api_const_options_values_PlayerImplementationValue::PROVIDER_BASED) {

            foreach ($embeddedPlayers as $embeddedPlayer) {

                /** @noinspection PhpUndefinedMethodInspection */
                if ($embeddedPlayer->getName() === $requestedEmbeddedPlayerName && $recognizingProvider !== null
                    && $recognizingProvider->getName() === $embeddedPlayer->getHandledProviderName()) {

                    //found it!
                    return $embeddedPlayer;
                }
            }
        }

        /**
         * None of the registered video providers recognize this video ID. Nothing we can do about that. This
         * should basically never happen.
         */
        if ($recognizingProvider === null) {

            return null;
        }

        /**
         * Do we have an embedded provider whose name exactly matches the provider? If so, let's use that. This
         * should be the common case.
         */
        foreach ($embeddedPlayers as $embeddedPlayer) {

            if ($embeddedPlayer->getName() === $recognizingProvider->getName()) {

                 return $embeddedPlayer;
            }
        }

        /**
         * Running out of options. See if we can find *any* player that can handle videos from this provider.
         */
        foreach ($embeddedPlayers as $embeddedPlayer) {

            if ($embeddedPlayer->getHandledProviderName() === $recognizingProvider->getName()) {

                return $embeddedPlayer;
            }
        }

        /**
         * None of the registered embedded players support the calculated provider. I give up.
         */
        return null;
    }

    private function _findProviderThatRecognizesVideoId($videoId)
    {
        $videoProviders = tubepress_impl_patterns_sl_ServiceLocator::getVideoProviders();

        foreach ($videoProviders as $videoProvider) {

            if ($videoProvider->recognizesVideoId($videoId)) {

                return $videoProvider;
            }
        }

        return null;
    }
}
